util.AddNetworkString("SFP_DARKRP:CreateJob")
util.AddNetworkString("SFP_DARKRP:RemoveJob")
util.AddNetworkString("SFP_DARKRP:LoadJobs")
util.AddNetworkString("SFP_DARKRP:RequestData")

local MODULE = {}
MODULE.Name = "Darkrp Integration"
MODULE.Enabled = true
MODULE.Version = "1.0"

function MODULE:CanLoad()
    return not (not DarkRP)
end

-- Called on all modules right after SFPortal Database is ready
-- There is no delay needed because the database module will do everything in a queue * pool
-- This also allows modules to have more control over the database, instead of relying on obscure functions inside the addon.
function MODULE:Migrations()
    local schema = [[
        SELECT * FROM information_schema.COLUMNS
        WHERE TABLE_SCHEMA = ':database:'
        AND TABLE_NAME = ':servertbl:'
        AND COLUMN_NAME = 'DarkRP_lastJobs'
        LIMIT 1;
    ]]
    schema = schema:Replace(
            ":database:",
             SFP.CONFIG("MySQL"):Get("Database")
        )
    schema = schema:Replace(
            ":servertbl:",
            SFP.CONFIG("General"):Get("serverTable")
        )

    local seed = [[
        ALTER TABLE `:servertbl:`
            ADD DarkRP_rpname varchar(50) DEFAULT "Unknown";
        ALTER TABLE `:servertbl:`
            ADD DarkRP_deaths int(255) DEFAULT "0";
        ALTER TABLE `:servertbl:`
            ADD  DarkRP_kills int(255)  DEFAULT "0";
        ALTER TABLE `:servertbl:`
            ADD  DarkRP_money bigint(255)  DEFAULT "0";
        ALTER TABLE `:servertbl:`
            ADD  DarkRP_arrests int(255)  DEFAULT "0";
        ALTER TABLE `:servertbl:`
            ADD  DarkRP_lastJobs varchar(50) DEFAULT "Unknown";
    ]]

    seed = seed:Replace(
        ":servertbl:",
        SFP.CONFIG("General"):Get("serverTable")
    )

    local schemaQuery = SFP.DB:Get():query(schema)

    function schemaQuery:onSuccess(data)
        if #data == 0 then
            Aw:LogTrace("DarkRP Integration was not detected on your database : Creating integration tables")
            local seedQuery = SFP.DB:Get():query(seed)

            function seedQuery:onSuccess(data)
                Aw:LogTrace("DarkRP Integration tables created : All good")
            end

            function seedQuery:onError(err)
                Aw:LogTrace("Failed creating DarkRP Integration tables : err \n%s\nQuery: %s", err, seed)
            end

            seedQuery:start()
        else
            Aw:LogTrace("DarkRP Integration detected : All good")
        end
        Aw:LogTrace("Loading DarkRP Custom jobs")
        MODULE:FetchJobs()
    end

    function schemaQuery:onError(err)
        Aw:LogTrace("Error during DarkRP Module migration: err \n%s\nQuery: %s", err, schema)
        return
    end

    schemaQuery:start()
end

-- This is called before DBReady, right when the module is registered
function MODULE:OnLoad()
    self.KillData = self.KillData or {}
    self.DeathData = self.DeathData or {}
    self.CustomJobs = self.CustomJobs or {}
    self.Players = self.Players or {}
    DarkRP.createCategory {
    	name = "custom",
    	categorises = "jobs",
    	startExpanded = true,
    	color = Color(0, 107, 0, 255),
    	sortOrder = 100,
    }

    hook.Add("PlayerInitialSpawn", "SFP:DarkRP:PlayerJoin", function(pPlayer)
        MODULE.Players[pPlayer:SteamID64()] = {}
    end)

    hook.Add("PlayerDeath", "SFP:DarkRP:PlayerDeath", function(victim, inflictor, attacker)
        if (attacker:IsPlayer()) then
            MODULE.KillData[attacker] = (MODULE.KillData[attacker] or 0) + 1
        end

        if (victim:IsPlayer()) then
            MODULE.DeathData[victim] = (MODULE.DeathData[victim] or 0) + 1
        end
    end)


    net.Receive("SFP_DARKRP:RequestData", function(xLen, pPlayer)
        Aw:LogTrace("Received requestData on server, seding table to compare:")
        MODULE:SendJob(MODULE.CustomJobs, pPlayer)
    end)
end

function MODULE:SendJob(xJobs, pPlayer)
    local isJobTable = type(xJobs) == "table"
    local targeted = IsValid(pPlayer)
    local jobCounts = isJobTable and table.Count(xJobs) or 1

    if isJobTable and jobCounts == 0 then
        Aw:LogTrace("Custom job table has not active jobs, not sending anything : All good")
        return
    end

    if not isJobTable then
        local command = xJobs
        xJobs = {
            MODULE.CustomJobs[command]
        }   
    end
    Aw:LogTrace("Sending %s custom job(s) to %s", jobCounts > 1 and "Multiple" or 1, targeted and pPlayer:Nick() or "everyone")

    net.Start("SFP_DARKRP:LoadJobs")

        net.WriteUInt(jobCounts, 16)

        for command, jobTable in pairs(xJobs) do
            net.WriteString( util.TableToJSON(jobTable.ammo) )
            net.WriteString(jobTable.category)
            net.WriteBool(jobTable.chief)
            net.WriteColor(jobTable.color)
            net.WriteString(jobTable.command)
            net.WriteString(jobTable.description)
            net.WriteUInt(jobTable.job_id, 16)
            net.WriteInt(jobTable.max, 16)
            net.WriteString( util.TableToJSON(jobTable.model) )
            net.WriteInt(jobTable.salary, 32)
            net.WriteString( util.TableToJSON(jobTable.weapons) )
            -- Custom check is a serverside function, skip
            net.WriteBool(jobTable.hasLicense)
            net.WriteBool(jobTable.medic)
            net.WriteBool(jobTable.cook)
            net.WriteBool(jobTable.mayor)
            net.WriteString(jobTable.name)
        end

    if targeted then
        net.Send(pPlayer)
    else
        net.Broadcast()
    end
end

function MODULE:Fetch()
  if (SFP.DB.STATUS != SFP.DBSTATUS.SUCCESS) then return end
  for _, pPlayer in pairs(player.GetAll()) do
    if (!SFP._tModules["DataSync"]) then Aw:LogTrace("DataSync Module not loaded") return end
    
    SFP._tModules["DataSync"].Players[pPlayer:SteamID64()]["DarkRP_rpname"] = {
        value = pPlayer:Nick()
    }

    SFP._tModules["DataSync"].Players[pPlayer:SteamID64()]["DarkRP_deaths"] = {
        value = MODULE.DeathData[pPlayer],
        isSum = true
    }

    SFP._tModules["DataSync"].Players[pPlayer:SteamID64()]["DarkRP_kills"] = {
        value = MODULE.KillData[pPlayer],
        isSum = true
    }

    SFP._tModules["DataSync"].Players[pPlayer:SteamID64()]["DarkRP_money"] = {
        value = pPlayer:getDarkRPVar("money")
    }

    SFP._tModules["DataSync"].Players[pPlayer:SteamID64()]["DarkRP_arrests"] = {
        value = pPlayer:getDarkRPVar("Arrested") and 1 or 0,
        isSum = true
    }

    SFP._tModules["DataSync"].Players[pPlayer:SteamID64()]["DarkRP_lastJobs"] = {
        value = pPlayer:getDarkRPVar("job")
    }
    MODULE.DeathData[pPlayer] = 0
    MODULE.KillData[pPlayer] = 0
  end
end

function MODULE:FetchJobs()

    local jobs = [[ SELECT * FROM sfp_darkrpjobs; ]]

    local jobsQuery = SFP.DB:Get():query(jobs)

    function jobsQuery:onSuccess(data)
        for _, row in pairs(data) do
            if (not row.status or row.status != 2) then
                Aw:LogTrace("Skipping disabled/inactive job : %s", row.job_id)
                continue
            end

            if (row.category != "" and row.category != null) then
                Aw:LogTrace("creating job panel category %s : Start good", row.category or "custom")

                local nonLiteralColor = Color(112, 0, 0)
                if (row.color) then
                    local colorTable = row.color.Explode(", ")
                    local r = colorTable[1] or 112
                    local g = colorTable[2] or 0
                    local b = colorTable[3] or 0
                    local a = colorTable[4] or 255
                    nonLiteralColor = Color(r, g, b, a)
                end

                DarkRP.createCategory {
                    name = row.category,
                    categorises = "jobs",
                    startExpanded = true,
                    color = nonLiteralColor,
                    sortOrder = 100,
                }        

                Aw:LogTrace("Category %s created : All Good", row.category or "custom")
            end
            
            Aw:LogTrace("Found job %s : %s Status: %s", row.name, row.job_id, row.status or 0)
            MODULE.CustomJobs[row.command] = {
                name = row.name,
                category = row.category or "custom",
                chief = row.chief == 1,
                color = Color(row.color_r or 112, row.color_g or 0, row.color_b or 0),
                command = row.command,
                description = row.description,
                job_id = row.job_id,
                max = row.max,
                model = util.JSONToTable(row.models),
                salary = row.salary,
                ammo = util.JSONToTable(row.ammo) or {},
                weapons = util.JSONToTable(row.weapons) or {},
                customCheck = function(pPlayer)
                    if row.public == 1 then
                        return true
                    end
                    return row.user_id == pPlayer:SteamID64()
                end,
                license = row.license == 1,
                admin = row.admin or 0,
                medic = row.medic == 1,
                cook = row.cook == 1,
                mayor = row.mayor == 1
            }

            DarkRP.createJob(row.name, MODULE.CustomJobs[row.command])
            Aw:LogTrace("Registered DarkRP Job on gamemode core : %s : Memory ref : %s", row.name,  MODULE.CustomJobs[row.command])
        end

    end

    function jobsQuery:onError(err)
        Aw:LogTrace("Error during DarkRP Jobs fetching : err\n%s\n%s", err, jobs)
        return
    end

    jobsQuery:start()
end

-- Called by a registered command, first argument is the commnad name, second is the argument table
function MODULE:LoadCustomJob(sCmnd, tArgs)
  if not tArgs[1] && tArgs[2] then
    Aw:LogTrace("Received job fetch from panel, but it's missing <action> or <id> argument")
    return
  end

  if (tArgs[1] =="add" or tArgs[1] == "update" or tArgs[1] == "remove") then
    local query = [[
        SELECT * FROM sfp_darkrpjobs
        WHERE job_id = :id:
    ]];

    

    query = query:Replace(":id:", tArgs[2])

    queryJob = SFP.DB.Get():query(query)

    function queryJob:onSuccess(data)
        local row = data[1]
        
        local t, id = DarkRP.getJobByCommand(row.command)
        if (id) then
                Aw:LogTrace("Tried to register job command: %s - But it's alredy assigned to %s", row.command, t.name .. " AKA " .. id)
        end
        Aw:LogTrace("Fetching job action requested by panel : %s - Command: %s", row.name or "Job non existent : Failed", row.command)

        if (tArgs[1] == "update") then
            
            local t, id = DarkRP.getJobByCommand(row.command)
            DarkRP.removeFromCategory(RPExtraTeams[id], "jobs")
            local tJob, index =	DarkRP.getJobByCommand(row.command)
            DarkRP.removeJob(index)
            
            MODULE.CustomJobs[row.command] = nil
            Aw:LogTrace("Job request was an update, removing the previous version of the job : teamID %s.", id)
            net.Start("SFP_DARKRP:RemoveJob")
                net.WriteString(row.command)
            net.Broadcast()
        end
               
        if (tArgs[1] == "remove") then
           Aw:LogTrace("Received request to remove JobCommand: %s - Start Good", row.command)
           
           local tRemovedJob = MODULE.CustomJobs[row.command]
           if !tRemovedJob then
			Aw:LogTrace("Job not found on cache : What the actual fuck? - JobCommand: %s - Attempt blocked", row.command)
            return
           end
           MODULE.CustomJobs[row.command] = nil
           Aw:LogTrace("Sending removeJob message to clients!")
           Aw:LogTrace("Removing job from server cache (SFP CACHE)")    
           local tJob, index =	DarkRP.getJobByCommand(row.command)
    	   DarkRP.removeJob(index)
           
           
           net.Start("SFP_DARKRP:RemoveJob")
          		net.WriteString(row.command)
           net.Broadcast()
           Aw:LogTrace("Sent removeJob message to clients!")
           return
        end
            
            
            MODULE.CustomJobs[row.command] = {
                name = row.name,
                category = row.category or "custom",
                chief = row.chief == 1,
                color = Color(row.color_r, row.color_g, row.color_b),
                command = row.command,
                description = row.description,
                job_id = row.job_id,
                max = row.max,
                model = util.JSONToTable(row.models),
                salary = row.salary,
                ammo = util.JSONToTable(row.ammo) or {},
                weapons = util.JSONToTable(row.weapons) or {},
                customCheck = function(pPlayer)
                    if row.public == 1 then
                        return true
                    end
                  return row.user_id == pPlayer:SteamID64()
                end,
                license = row.license == 1,
                admin = row.admin  or 0,
                medic = row.medic == 1,
                cook = row.cook == 1,
                mayor = row.mayor == 1
            }
            DarkRP.createJob(row.name, MODULE.CustomJobs[row.command])
            MODULE:SendJob(row.command)
            Aw:LogTrace("Registered/Updated DarkRP Job on gamemode core : %s : Memory ref : %s", row.name,  MODULE.CustomJobs[row.command])
       end


    function queryJob:onError(err)
        Aw:LogTrace("Panel requested a job, but game was unable to fetch it : err\n%s", err)
    end

    queryJob:start()

    else
        Aw:LogTrace("Received Jobs command, but the action argument doesn´t match with any valid action : %s", tArgs[1])
    end
end

SFP:RegisterModule(MODULE)
SFP:RegisterCommand("darkrpjobs", MODULE.LoadCustomJob, MODULE)
